/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.CascadingValueProvider;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldValueProvider;
import com.inet.report.Fields;
import com.inet.report.PromptField;
import com.inet.report.SQLValueProvider;
import com.inet.report.SortField;

/**
 * This example shows you how to use dynamic prompt fields with RDC. Dynamic Prompts are prompts whose default values
 * are generated dynamically at run-time, rather than set to a number of static values which never change. To create a
 * dynamic prompt, you must simply call
 * {@link PromptField#setDefaultValueProvider(com.inet.report.DynamicValueProvider)}. There are currently three
 * DefaultValueProvider implementations which are included with i-net Clear Reports - {@link FieldValueProvider},
 * {@link SQLValueProvider}, and {@link CascadingValueProvider}, however you could also easily write your own.
 */
public class DynamicPromptsSample extends RDCSample {

    //change this before running the example if the location of the Report
    //"mainrep.rpt" which comes
    //with the rdc-samples has changed
    static private final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Add a dynamic parameter field to an existing report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = new Engine( exportFmt );
            //set the name of the report file that contains the report
            URL reportURL = getClass().getResource( '/' + MAINREPORT );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( MAINREPORT );
            }
            Fields fields = eng.getFields();

            PromptField pField = null;
            //add a prompt field to the fields
            pField = fields.addPromptField( "country", "Customers from which country?", Field.STRING );

            // This creates a "dynamic value provider" which fetches the prompt's default values
            // at run-time.
            FieldValueProvider provider = new FieldValueProvider( fields.getDatabaseField( "Customers.Country" ), null, SortField.DESCENDING_ORDER );

            // this will cause the prompt's default values to be dynamically fetched at
            // run time from the given database field
            pField.setDefaultValueProvider( provider );

            // change the selection formula so that only the records are shown where the country is
            // the same as the country chosen by the user.
            eng.setSF( "{Customers.Country} = {?country}" );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new DynamicPromptsSample().initUI();
    }
}
